<?php
/* --------------------------------------------------------------
   UpdateParcelServicesAction.php 2020-04-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\ParcelService\Actions;

use Gambio\Admin\ParcelService\Interfaces\ParcelServiceService;
use Gambio\Admin\ParcelService\Model\ParcelService;
use Gambio\Admin\ParcelService\Model\ParcelServices;
use Gambio\Api\ParcelService\ParcelServiceApiRequestValidator;
use InvalidArgumentException;
use Slim\Http\Response;
use Slim\Http\ServerRequest as Request;

/**
 * Class UpdateParcelServicesAction
 *
 * @package Gambio\Api\ParcelService\Actions
 */
class UpdateParcelServicesAction
{
    /**
     * @var ParcelServiceApiRequestValidator
     */
    private $requestValidator;
    
    /**
     * @var ParcelServiceService
     */
    private $service;
    
    
    /**
     * UpdateParcelServicesAction constructor.
     *
     * @param ParcelServiceApiRequestValidator $requestValidator
     * @param ParcelServiceService             $service
     */
    public function __construct(
        ParcelServiceApiRequestValidator $requestValidator,
        ParcelServiceService $service
    ) {
        $this->requestValidator = $requestValidator;
        $this->service          = $service;
    }
    
    
    /**
     * @param Request  $request
     * @param Response $response
     * @param array    $args
     *
     * @return Response
     */
    public function __invoke(Request $request, Response $response, array $args): Response
    {
        $referencedErrors = $this->requestValidator->validatePutRequestBody($request->getParsedBody());
        if (count($referencedErrors) > 0) {
            return $response->withStatus(400)->withJson(['errors' => $referencedErrors]);
        }
        
        $parcelServices = [];
        foreach ($request->getParsedBody() as $reference => $details) {
            try {
                $parcelServices[$reference] = ParcelService::createFromArray($details);
            } catch (InvalidArgumentException $exception) {
                $referencedErrors[$reference][] = $exception->getMessage();
            }
        }
        if (count($referencedErrors) > 0) {
            return $response->withStatus(422)->withJson(['errors' => $referencedErrors]);
        }
        
        $this->service->storeParcelServices(ParcelServices::createWithReferences($parcelServices));
        
        return $response->withStatus(204);
    }
}